<?php

namespace GSPIN\ShortcodeBuilder;

// if direct access than exit the file.
defined('ABSPATH') || exit;

class Manager {

    public function __construct() {
        add_action( 'wp_ajax_gspin_update_shortcode', array( $this, 'updateShortcode' ) );
        add_action( 'wp_ajax_gspin_create_shortcode', array( $this, 'createShortcode' ) );
        add_action( 'wp_ajax_gspin_clone_shortcode', array( $this, 'cloneShortcode' ) );
        add_action( 'wp_ajax_gspin_get_shortcode', array( $this, 'getShortcode' ) );
        add_action( 'wp_ajax_gspin_get_shortcodes', array( $this, 'getShortcodes' ) );
        add_action( 'wp_ajax_gspin_delete_shortcodes', array( $this, 'deleteShortcode' ) );
        add_action( 'wp_ajax_gspin_temp_save_shortcode_settings', array( $this, 'saveTempShortcodeSettings' ) );
    }

    /**
     * Returns the shortcode default settings.
     * 
     * @since  2.0.12
     * @return array The predefined default settings for each shortcode.
     */
    public function getShortcodeDefaultSettings( $atts = [] ) {
        return gsPinBuilder()->manager->getShortcodeAttributes( $atts );
    }

    /**
     * Returns the shortcode by given id.
     * 
     * @since  2.0.12
     * 
     * @param mixed $shortcode_id The shortcode id.
     * @param bool  $is_ajax       Ajax status.
     * 
     * @return array|JSON The shortcode.
     */
    public function getShortcodeById( $shortcode_id, $is_ajax = false ) {
        if ( empty( $shortcode_id ) ) {
            if ( $is_ajax ) {
                wp_send_json_error(
                    __('Shortcode ID missing', 'gs-pinterest'),
                    400
                );
            }

            return false;
        }

        $wpdb      = gsPinBuilder()->data->getWpdb();
        $tableName = gspin()->db->getShortcodesTable();
        $shortcode = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$tableName} WHERE id = %d LIMIT 1", absint($shortcode_id) ), ARRAY_A );

        if ( $shortcode ) {
            $shortcode["shortcode_settings"] = json_decode( $shortcode["shortcode_settings"], true );

            if ( $is_ajax ) {
                wp_send_json_success( $shortcode );
            }

            return $shortcode;
        }

        if ( $is_ajax ) {
            wp_send_json_error( __('No shortcode found', 'gs-pinterest'), 404 );
        }

        return false;
    }

    /**
     * Ajax endpoint for get shortcodes.
     * 
     * @since  2.0.12
     * @return JSON The response as json object.
     */
    public function getShortcodes() {
        return gsPinBuilder()->manager->fetchShortcodes( null, wp_doing_ajax() );
    }

    /**
     * Ajax endpoint for get shortcodes.
     * 
     * @since  2.0.12
     * @return JSON The response as json object.
     */
    public function getShortcodesAsList() {
        return gsPinBuilder()->manager->fetchShortcodes( null, false );
    }

    /**
     * Fetch shortcodes by given shortcode ids.
     * 
     * @since  2.0.12
     * 
     * @param mixed $shortcode_ids Shortcode ids.
     * @param bool  $is_ajax       Ajax status.
     * @param bool  $minimal        Shortcode minimal result.
     * 
     * @return array|json Fetched shortcodes.
     */
    public function fetchShortcodes( $shortcode_ids = [], $is_ajax = false, $minimal = false ) {
        $wpdb      = gsPinBuilder()->data->getWpdb();
        $fields    = $minimal ? 'id, shortcode_name' : '*';
        $tableName = gspin()->db->getShortcodesTable();

        if ( empty( $shortcode_ids ) ) {
            $shortcodes = $wpdb->get_results( "SELECT {$fields} FROM {$tableName} ORDER BY id DESC", ARRAY_A );
        } else {
            $how_many     = count( $shortcode_ids );
            $placeholders = array_fill( 0, $how_many, '%d' );
            $format       = implode( ', ', $placeholders );
            $query        = "SELECT {$fields} FROM {$tableName} WHERE id IN($format)";
            $shortcodes   = $wpdb->get_results( $wpdb->prepare( $query, $shortcode_ids ), ARRAY_A );
        }

        // check for database error
        if ( gsPinBuilder()->data->error() ) {
            wp_send_json_error( sprintf( __( 'Database Error: %s' ), $wpdb->last_error ) );
        }

        if ( $is_ajax ) {
            wp_send_json_success( $shortcodes );
        }

        return $shortcodes;
    }

    /**
     * Save shortcode temporary default settings.
     * 
     * @since  2.0.12
     * 
     * @return saved status.
     */
    public function saveTempShortcodeSettings() {
        if ( ! check_admin_referer( '_gspin_temp_save_shortcode_settings_gs_' ) || ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( __('Unauthorised Request', 'gs-pinterest'), 401 );
        }


        $tempShortcode      = [];
        $temp_key           = isset( $_POST[ 'temp_key' ] ) ? $_POST[ 'temp_key' ] : null;
        $tempShortcode['shortcode_settings'] = isset( $_POST[ 'shortcode_settings' ] ) ? $_POST[ 'shortcode_settings' ] : null;
        $tempShortcode['userid']             = isset( $_POST[ 'userid' ] ) ? $_POST[ 'userid' ] : null;
        $tempShortcode['board_name']         = isset( $_POST[ 'board_name' ] ) ? $_POST[ 'board_name' ] : null;
        $tempShortcode['count']              = isset( $_POST[ 'count' ] ) ? $_POST[ 'count' ] : null;
        $tempShortcode['show_pin_title']     = isset( $_POST['show_pin_title'] ) ? $_POST['show_pin_title'] : 0;

        if ( empty( $temp_key ) ) {
            wp_send_json_error( __('No temp key provided', 'gs-pinterest'), 400 );
        }

        if ( empty( $tempShortcode['shortcode_settings'] ) ) {
            wp_send_json_error( __('No temp settings provided', 'gs-pinterest'), 400 );
        }

        delete_transient( $temp_key );
        set_transient( $temp_key, $tempShortcode, 86400 ); // save the transient for 1 day

        wp_send_json_success( __( 'Temp data saved', 'gs-pinterest' ) );
    }

    /**
     * Get shortcode temporary settings by id.
     * 
     * @since  2.0.12
     * @return array  Shortcode settings.
     */
    public function getTempSettings( $id ) {
        return get_transient( $id );
    }

    /**
     * Validate given shortcode settings.
     * 
     * @since  2.0.12
     * 
     * @param  array $settings
     * @return array Shortcode settings.
     */
    public function validateSettings( $settings ) {
        return (array) $settings;
    }

    /**
     * Ajax endpoint for deleting shortcode.
     * 
     * @since  2.0.12
     * @return JSON The response as json object.
     */
    public function deleteShortcode() {
        if ( ! check_admin_referer( '_gspin_delete_shortcodes_gs_' ) || ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( __( 'Unauthorised Request', 'gs-pinterest' ), 401 );
        }

        $ids = isset( $_POST['ids'] ) ? $_POST['ids'] : null;

        if ( empty( $ids ) ) {
            wp_send_json_error( __( 'No shortcode ids provided', 'gs-pinterest' ), 400 );
        }

        $wpdb  = gsPinBuilder()->data->getWpdb();
        $count = count( $ids );

        $ids = implode( ',', array_map( 'absint', $ids ) );
        $tableName = gspin()->db->getShortcodesTable();
        $wpdb->query( "DELETE FROM {$tableName} WHERE ID IN($ids)" );

        if ( gsPinBuilder()->data->error() ) {
            wp_send_json_error( sprintf( __( 'Database Error: %s' ), $wpdb->last_error ), 500 );
        }

        $m = _n( "Shortcode has been deleted", "Shortcodes have been deleted", $count, 'gs-pinterest' ) ;
        wp_send_json_success( [ 'message' => $m ] );
    }

    /**
     * Ajax endpoint for update shortcode.
     * 
     * @since  2.0.12
     * @return int The numbers of row affected.
     */
    public function updateShortcode() {
        if ( ! check_admin_referer( '_gspin_update_shortcode_gs_' ) || ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( __( 'Unauthorised Request', 'gs-pinterest' ), 401 );
        }

        $shortcode_id = ! empty( $_POST[ 'id' ] ) ? absint( $_POST[ 'id' ] ) : null;

        if ( empty( $shortcode_id ) ) {
            wp_send_json_error( __('Shortcode ID missing', 'gs-pinterest'), 400 );
        }

        $shortcode    = gsPinBuilder()->manager->getShortcodeById( $shortcode_id, false );
    
        if ( empty( $shortcode ) ) {
            wp_send_json_error( __('No shortcode found to update', 'gs-pinterest'), 404 );
        }
    
        $shortcode_name      = ! empty( $_POST['shortcode_name'] ) ? sanitize_text_field( $_POST['shortcode_name'] ) : sanitize_text_field( $shortcode[ 'shortcode_name' ] );
        $shortcode_settings  = ! empty( $_POST['shortcode_settings'] ) ? $_POST['shortcode_settings'] : $shortcode['shortcode_settings'];
        $userid         = ! empty( $_POST['userid'] ) ? sanitize_text_field( $_POST['userid'] ) : '';
        $count          = ! empty( $_POST['count'] ) ? absint( $_POST['count'] ) : '';
        $board_name     = ! empty( $_POST['board_name'] ) ? $_POST['board_name'] : '';
        $show_pin_title = ! empty( $_POST['show_pin_title'] ) && $_POST['show_pin_title'] == "true" ? 1 : 0;
    
        $shortcode_settings = gsPinBuilder()->manager->validateSettings( $shortcode_settings );
        $tableName          = gspin()->db->getShortcodesTable();
        $wpdb               = gsPinBuilder()->data->getWpdb();
    
        $data = array(
            "shortcode_name"     => $shortcode_name,
            "shortcode_settings" => json_encode( $shortcode_settings ),
            "userid"             => $userid,
            "board_name"         => $board_name,
            "count"              => $count,
            'show_pin_title'     => $show_pin_title
        );
    
        $updateId = $wpdb->update(
            $tableName,
            $data,
            array(
                'id' => $shortcode_id
            ), 
            gsPinBuilder()->data->getColumns()
        );
    
        if ( gsPinBuilder()->data->error() ) {
            wp_send_json_error(
                sprintf( __( 'Database Error: %1$s', 'gs-pinterest' ),
                $wpdb->last_error
            ), 500 );
        }
    
        wp_send_json_success( array(
            'message'      => __( 'Shortcode updated', 'gs-pinterest' ),
            'shortcode_id' => $updateId
        ));
    }

    /**
     * Ajax endpoint for create shortcode.
     * 
     * @since  2.0.12
     * @return json WP json response.
     */
    public function createShortcode() {
        if ( ! check_admin_referer( '_gspin_create_shortcode_gs_' ) || ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( __( 'Unauthorised Request', 'gs-pinterest' ), 401 );
        }

        $shortcode_name     = ! empty( $_POST[ 'shortcode_name' ] ) ? sanitize_text_field( $_POST[ 'shortcode_name' ] ) : __( 'Untitled', 'gs-pinterest' );
        $shortcode_settings = ! empty( $_POST[ 'shortcode_settings' ] ) ? $_POST[ 'shortcode_settings' ] : '';
        $userid      = ! empty( $_POST['userid'] ) ? sanitize_text_field( $_POST['userid'] ) : '';
        $board_name      = ! empty( $_POST['board_name'] ) ? sanitize_text_field( $_POST['board_name'] ) : '';
        $count = ! empty( $_POST['count'] ) ? absint( $_POST['count'] ) : '';
        $show_pin_title = ! empty( $_POST['show_pin_title'] ) ?  $_POST['show_pin_title'] : false;

        if ( empty( $shortcode_settings ) || ! is_array( $shortcode_settings ) ) {
            wp_send_json_error( __( 'Please configure the settings properly', 'gs-pinterest' ), 206 );
        }

        $shortcode_settings = gsPinBuilder()->manager->validateSettings( $shortcode_settings );

        $wpdb               = gsPinBuilder()->data->getWpdb();
        $tableName          = gspin()->db->getShortcodesTable();

        $data = array(
            "shortcode_name"     => $shortcode_name,
            "shortcode_settings" => json_encode( $shortcode_settings ),
            "userid"             => $userid,
            "board_name"         => $board_name,
            "count"              => $count,
            'show_pin_title'    => $show_pin_title
        );

        $wpdb->insert( $tableName, $data, gsPinBuilder()->data->getColumns() );

        // check for database error
        if ( gsPinBuilder()->data->error() ) {
            wp_send_json_error( sprintf( __( 'Database Error: %s' ), $wpdb->last_error ), 500 );
        }

        // send success response with inserted id
        wp_send_json_success( array(
            'message'      => __( 'Shortcode created successfully', 'gs-pinterest' ),
            'shortcode_id' => $wpdb->insert_id
        ));
    }

    /**
     * Ajax endpoint for clone a shortcode.
     * 
     * @since  2.0.12
     * @return json WP json response.
     */
    public function cloneShortcode() {
        if ( ! check_admin_referer( '_gspin_clone_shortcode_gs_' ) || ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( __( 'Unauthorised Request', 'gs-pinterest' ), 401 );
        }

        $clone_id  = ! empty( $_POST[ 'clone_id' ] ) ? absint( $_POST[ 'clone_id' ] ) : '';

        if ( empty( $clone_id ) ) {
            wp_send_json_error( __( 'Clone Id not provided', 'gs-pinterest' ), 400 );
        }

        $clone_shortcode = gsPinBuilder()->manager->getShortcodeById( $clone_id, false );

        if ( empty( $clone_shortcode ) ) {
            wp_send_json_error( __( 'No shortcode found to clone.', 'gs-pinterest' ), 404 );
        }

        $shortcode_settings = $clone_shortcode[ 'shortcode_settings' ];
        $shortcode_name     = $clone_shortcode[ 'shortcode_name' ] .' '. __( '- Cloned', 'gs-pinterest' );
        $shortcode_settings = gsPinBuilder()->manager->validateSettings( $shortcode_settings );

        $wpdb      = gsPinBuilder()->data->getWpdb();
        $tableName = gspin()->db->getShortcodesTable();

        $data = array(
            "shortcode_name"     => $shortcode_name,
            "shortcode_settings" => json_encode($shortcode_settings)
        );

        $wpdb->insert(
            $tableName,
            $data,
            gsPinBuilder()->data->getColumns()
        );

        if ( gsPinBuilder()->data->error() ) {
            wp_send_json_error( sprintf( __( 'Database Error: %s' ), $wpdb->last_error ), 500 );
        }

        // Get the cloned shortcode
        $shotcode = gsPinBuilder()->manager->getShortcodeById( $wpdb->insert_id, false );

        // send success response with inserted id
        wp_send_json_success( array(
            'message'   => __( 'Shortcode cloned successfully', 'gs-pinterest' ),
            'shortcode' => $shotcode,
        ));
    }

    /**
     * Ajax endpoint for get a shortcode.
     * 
     * @since  1.10.14
     * @return void
     */
    public function getShortcode() {
        $shortcode_id = ! empty( $_GET['id']) ? absint( $_GET['id'] ) : null;
        return gsPinBuilder()->manager->getShortcodeById( $shortcode_id, wp_doing_ajax() );
    }

    /**
     * Returns all required shortcode attributes for this plugin.
     * 
     * @since  1.10.14
     * @return array   Shortcode attributes.
     */
    public function getShortcodeAttributes( $atts = [] ) {
        $gs_pin_user     = gspin()->helpers->getOption( 'gs_pin_user', '' );
		$gs_pin_board    = gspin()->helpers->getOption( 'gs_pin_board', '' );
		$gs_tot_pins     = gspin()->helpers->getOption( 'gs_tot_pins', '' );
		$gs_pin_theme    = gspin()->helpers->getOption( 'gs_pin_theme', 'gs_pin_theme1' );
		$gs_pins_col     = gspin()->helpers->getOption( 'gs_pins_col', 25 );

		return shortcode_atts(
            array(
                'id'                      => '',
                'user'                    => $gs_pin_user,
                'board'                   => $gs_pin_board,
                'count'                   => $gs_tot_pins,
                'theme'                   => $gs_pin_theme,
                'cols'                    => $gs_pins_col,
                'link_target'             => '_blank',
                'columns'                 => '4',
                'columns_mobile'          => '12',
                'columns_mobile_portrait' => '12',
                'columns_tablet'          => '6'
        	), 
			$atts
		);
    }
}