<?php

namespace GSPIN;

// if direct access than exit the file.
defined('ABSPATH') || exit;

class Helpers {
    /**
     * Retrives free plugin list based on the author.
     * 
     * @since  2.0.8
     * @return array Plugins collection.
     */
    public function getFreePlugins() {
        include( ABSPATH . "wp-admin/includes/plugin-install.php" );

        global $tabs, $tab, $paged, $type, $term;
        $tabs     = array();
        $tab      = "search";
        $per_page = 20;

        $args = array (
            "author" 	=> "samdani",
            "page" 		=> $paged,
            "per_page" 	=> $per_page,
            "fields" 	=> array( "last_updated" => true, "downloaded" => true, "icons" => true ),
            "locale" 	=> get_locale(),
        );

        $arges 	= apply_filters( "install_plugins_table_api_args_$tab", $args );
        $api 	= plugins_api( "query_plugins", $arges );
        $items 	= $api->plugins;

        return $items;
    }

    /**
     * Generates star ratings.
     * 
     * @since  2.0.8
     * @return html Generated rating stars html markup.
     */
    public function wp_star_rating( $args = array() ) {
        $defaults = array(
            'rating' => 0,
            'type'   => 'rating',
            'number' => 0,
        );

        $r = wp_parse_args( $args, $defaults );

        // Non-english decimal places when the $rating is coming from a string
        $rating = str_replace( ',', '.', $r['rating'] );

        // Convert Percentage to star rating, 0..5 in .5 increments
        if ( 'percent' == $r['type'] ) {
            $rating = round( $rating / 10, 0 ) / 2;
        }

        // Calculate the number of each type of star needed
        $full_stars  = floor( $rating );
        $half_stars  = ceil( $rating - $full_stars );
        $empty_stars = 5 -  $full_stars - $half_stars;

        if ( $r['number'] ) {
            $format = _n( '%1$s rating based on %2$s rating', '%1$s rating based on %2$s ratings', $r['number'] );
            $title = sprintf( $format, number_format_i18n( $rating, 1 ), number_format_i18n( $r['number'] ) );
        } else {
            $title = sprintf( __( '%s rating' ), number_format_i18n( $rating, 1 ) );
        }

        echo '<div class="star-rating" title="' . esc_attr( $title ) . '">';
            echo '<span class="screen-reader-text">' . $title . '</span>';
            echo str_repeat( '<div class="star star-full"></div>', $full_stars );
            echo str_repeat( '<div class="star star-half"></div>', $half_stars );
            echo str_repeat( '<div class="star star-empty"></div>', $empty_stars);
        echo '</div>';
    }

    /**
     * Checks for if the current shortcode is from Shortcode Builder.
     * 
     * @since 2.0.8
     * 
     * @param  array $atts Shortcode attributes.
     * @return bool       Boolean value based on the id.
     */
    public function isFromShortcodeBuilder( $atts ) {
        return ! empty( $atts[ 'id' ] );
    }

    /**
     * Builds url for fetching pins.
     * 
     * @since 2.0.8
     * 
     * @param string  $username Pinterest username.
     * @param string  $board    Pinterest board name.
     * @return string          Pinterest pins url.
     */
    public function buildUrl( $username, $board ) {
        if ( empty( $username ) ) {
			return __( 'Username is not valid', 'gs-pinterest' );
		}

        if( ! empty( $board ) ) {  
            $board    = str_replace( ' ', '-', $board );
            return 'https://pinterest.com/' . $username . '/' . $board . '.rss';
        }

        return 'https://pinterest.com/' . $username . '/feed.rss'; 
    }

    /**
     * Fetch pins from given url.
     * 
     * @since 2.0.8
     * 
     * @param  string $feedUrl Pinterest pins url.
     * @param  int    $count   The number of pins to fetch.
     * @return obj             Fetched pins.
     */
    public function fetchPins( $feedUrl, $count ) {
        $pins = " ";

        if ( ! empty( $username ) ) {
			return $pins;
		}
        
		$pins = fetch_feed( $feedUrl );
        if ( is_wp_error( $pins ) ) {
            return __( 'Something went wrong, No Pins found.!', 'gs-pinterest' );
        }

        $max  = $pins->get_item_quantity( $count );
        $pins = $pins->get_items( 0, $max );
        return $pins;
    }

    /**
     * Returns each item columns.
     * 
     * @since 2.0.8
     * 
     * @param string $desktop         The option name.
     * @param string $tablet          Settings section name.
     * @param string $mobile_portrait The value to save.
     * @param string $mobile          The value to save.
     * 
     * @return string Item columns.
     */
    public function getColumnClasses( $desktop = '3', $tablet = '4', $mobile_portrait = '6', $mobile = '12' ) {
        return sprintf('gs-col-lg-%s gs-col-md-%s gs-col-sm-%s gs-col-xs-%s', $desktop, $tablet, $mobile_portrait, $mobile );
    }

    /**
     * Generates styles for each pin block.
     * 
     * @since  2.0.8
     * @return css Generated styles.
     */
    public function generateStyles( $gutter ) {
        $margin = $gutter / 2;
        $css = "<style>
                .gs-pin-details, .gs-pin-pop {
                    margin: {$margin}px;
                }
            </style>";

        return $css;
	}

    /**
     * Checks for is plugin pro version active.
     * 
     * @since 2.0.8
     */
    public function isProActive() {
        return defined( 'GSPIN_PRO_VERSION' );
    }

    /**
     * Generate masonry scripts for each shortcode.
     * 
     * @since  2.0.8
     * @return mixed masonry js.
     */
    public function generateScripts() {
        $js = "<script>
                (function($){
                    $(document).ready(function(){
                        var wrap = $('.gs-pins');
                        var grid = wrap.masonry({
                            itemSelector    : '.gs-single-pin',
                            columnWidth     : '.gs-single-pin',
                            percentPosition : true
                        });

                        // layout Masonry after each image loads
                        grid.imagesLoaded().progress( function() {
                            grid.masonry('layout');
                        });
                    });
                })(jQuery);
            </script>";
        return $js;
    }

    /**
     * Retrives option from settings options.
     * 
     * @since  2.0.8
     * @return mixed option value.
     */
    public function getOption( $option, $default = '' ) {
        $options = get_option( 'gs_pinterest_settings' );
        if ( isset( $options[ $option ] ) ) {
            return $options[ $option ];
        }
        return $default;
    }

    /**
     * Renders boards widget.
     * 
     * @since 2.0.8
     */
    public function boardsWidget( $gspin_url, $gspin_label, $gspin_size, $gspin_cstm_sizes, $gspin_action ) {
        // Pinterest default "Square" size
        $gspin_scale_width  = 80;
        $gspin_scale_height = 320;
        $gspin_board_width  = 400;
        
        // Sidebar size
        if( $gspin_size == 'sidebar' ) {
            $gspin_scale_width  = 60;
            $gspin_scale_height = 800;
            $gspin_board_width  = 150;
        }
        
        // Header size
        if( $gspin_size == 'header' ) {
            $gspin_scale_width  = 115;
            $gspin_scale_height = 120;
            $gspin_board_width  = 900;
        }
        
        // Custom size
        if( $gspin_size == 'custom' ) {
            // Can't be blank & MUST need greater than minimum value by Pinterest to get output.
            $gspin_scale_width  = ( $gspin_cstm_sizes['width'] >= 60 ? $gspin_cstm_sizes['width'] : '' );
            $gspin_scale_height = ( $gspin_cstm_sizes['height'] >= 60 ? $gspin_cstm_sizes['height'] : '' );
            $gspin_board_width  = ( $gspin_cstm_sizes['board_width'] >= 130 ? $gspin_cstm_sizes['board_width'] : '' );
        }
        
        if( $gspin_action == 'embedUser' ) {
            $gspin_url = "http://www.pinterest.com/" . $gspin_url;
        }
        
        $output  = '<a data-pin-do="' . $gspin_action . '"';
        $output .= 'href="' . esc_attr( $gspin_url ) . '"';
        $output .= ( ! empty( $gspin_scale_width ) ? 'data-pin-scale-width="' . $gspin_scale_width . '"' : '' );
        $output .= ( ! empty( $gspin_scale_height ) ? 'data-pin-scale-height="' . $gspin_scale_height . '"' : '' );
        // if the board_width is empty then it has been set to 'auto' so we need to leave the data-pin-board-width attribute completely out
        $output .= ( ! empty( $gspin_board_width ) ? 'data-pin-board-width="' . $gspin_board_width . '"' : '' );
        $output .= '>' . $gspin_label . '</a>';
        
        return $output;
    }

    /**
     * Returns shortcodes as select option.
     * 
     * @since 2.0.8
     * @param boolean $options If we want's the value as options.
     * @param boolean $default If we want's the value as the default option.
     * 
     * @return mixed Options array or the default value.
     */
    public function getShortcodeAsOptions() {
        $shortcodes = gsPinBuilder()->manager->getShortcodesAsList();
        if ( empty( $shortcodes ) ) {
            return;
        }

        return array_combine(
            wp_list_pluck( gsPinBuilder()->manager->getShortcodesAsList(), 'id' ),
            wp_list_pluck( gsPinBuilder()->manager->getShortcodesAsList(), 'shortcode_name' )
        );

    }

    /**
     * Returns shortcodes as select option just for visual composer.
     * 
     * @since 2.0.8
     * @return array Options array
     */
    public function getVcShortcodeOptions() {
        $shortcodes = $this->getShortcodeAsOptions();

        if ( empty( $shortcodes ) ) {
            return;
        }

        return array_flip( $shortcodes );
    }

    /**
     * Returns default item from shortcode list.
     * 
     * @since 2.0.8
     */
    public function getDefaultOption() {
        $shortcodes = gsPinBuilder()->manager->getShortcodesAsList();

        if ( ! empty( $shortcodes ) ) {
            return $shortcodes[0]['id'];
        }

        return '';
    }
}